local mq = require('mq')
local config = require('config')
local utils = {}
local lastAbilityUseTime = os.time()
local abilityCooldown = 30
local lastMacroCheckTime = 0
local macroCheckInterval = 10
local lastRoleSetTime = 0
local roleSetDelay = 2

function utils.checkMacroStatus(config)
    if config.checkMacroStatus then
        local currentTime = os.time()

        if not config.checkStarted and currentTime - lastRoleSetTime >= roleSetDelay then
            config.checkStarted = true
        end

        if config.checkStarted then
            local currentTime = os.time()
            if currentTime - lastMacroCheckTime >= macroCheckInterval then
                lastMacroCheckTime = currentTime

                local macroName = mq.TLO.Macro.Name()
                local macroPaused = mq.TLO.Macro.Paused()

                if macroName == nil or macroName == '' then
                    config.isKissAssistRunning = false
                    config.iniLoaded = false
                    config.selectedPath = ""
                    config.current_item = 1
                    config.isMacroPaused = false -- Reset the paused state
                    config.checkMacroStatus = false
                    config.macroStarted = false
                    print("KISSASSIST IS NOT RUNNING! Resetting Role to 'Not Set' and requiring INI reload.")
                else
                    config.isKissAssistRunning = true
                    config.isMacroPaused = (macroPaused == true)
                end
            end
        end
    end
end
function utils.parseIniFile(filePath)
    local settings = {}
    local file, err = io.open(filePath, "r")
    if not file then
        return nil, "Unable to open file: " .. err
    end

    for line in file:lines() do
        if not line:match("^%s*[;#]") and line:match("%S") then
            local key, value = line:match("^%s*([^=]+)%s*=%s*(.+)%s*$")
            if key and value then
                settings[key:match("^%s*(.-)%s*$")] = value:match("^%s*(.-)%s*$")
            end
        end
    end

    file:close()
    return settings
end

function utils.openFileDialog()
    local tempFilePath = "selectedFilePath.txt"
    local command =
        'powershell -Command "Add-Type -AssemblyName System.Windows.Forms; $dialog = New-Object System.Windows.Forms.OpenFileDialog; $null = $dialog.ShowDialog(); $dialog.FileName > ' ..
        tempFilePath .. '"'
    os.execute(command)

    local file = io.open(tempFilePath, "r")
    if file then
        local selectedPath = file:read("*a"):match("^%s*(.-)%s*$")
        file:close()
        os.remove(tempFilePath)

        selectedPath = selectedPath:gsub("[%z\1-\31\128-\255]", ""):gsub("\\", "\\\\")
        return selectedPath
    else
        return nil
    end
end

function findSettingIgnoreCase(settings, searchKey)
    local searchKeyLower = searchKey:lower()
    for key, value in pairs(settings) do
        if key:lower() == searchKeyLower then
            return value
        end
    end
    return nil
end

function utils.validateNumericInput(input)
    local number = tonumber(input)
    return number and number >= 0 and number <= 999 and tostring(number) or ""
end

function utils.saveSettingsToIniFile(filePath, config)
    local setPullLevel = config.setPullLevelMin .. "|" .. config.setPullLevelMax
    local managedSettings = {
        MaxRadius = config.setRadius and "MaxRadius=" .. config.setRadius or nil,
        PullArcWidth = config.setArc and "PullArcWidth=" .. config.setArc or nil,
        MedStart = config.setMed and "MedStart=" .. config.setMed or nil,
        AssistAt = config.setAssist and "AssistAt=" .. config.setAssist or nil,
        MaxZRange = config.setZrange and "MaxZRange=" .. config.setZrange or nil,
        MezRadius = config.setMezradius and "MezRadius=" .. config.setMezradius or nil,
        MezStopHPs = config.setStophp and "MezStopHPs=" .. config.setStophp or nil,
        ChainPullHP = config.ChainPullHP and "ChainPullHP=" .. config.ChainPullHP or nil,
        PullWait = config.PullWait and "PullWait=" .. config.PullWait or nil,
        PullLevel = setPullLevel and "PullLevel=" .. setPullLevel or nil,
        MezMinLevel = config.MezMinLevel and "MezMinLevel=" .. config.MezMinLevel or nil,
        MezMaxLevel = config.MezMaxLevel and "MezMaxLevel=" .. config.MezMaxLevel or nil,
        PetAssistAt = config.PetAssistAt and "PetAssistAt=" .. config.PetAssistAt or nil,
        AERadius = config.AERadius and "AERadius=" .. config.AERadius or nil,
        StickHow = config.stickHow and "StickHow=" .. config.stickHow or nil,
    }

    local lines = {}
    local file, err = io.open(filePath, "r")
    if file then
        for line in file:lines() do
            local key = line:match("^%s*([^=]+)%s*=")
            if key and managedSettings[key] then
                table.insert(lines, managedSettings[key])
            else
                table.insert(lines, line)
            end
        end
        file:close()
    else
        print("Error reading file for updating: " .. (err or "Unknown error"))
        return
    end

    file, err = io.open(filePath, "w")
    if file then
        for _, line in ipairs(lines) do
            if line then
                file:write(line .. "\n")
            end
        end
        file:close()
    else
        print("Error writing file: " .. (err or "Unknown error"))
    end
end

function utils.contains(table, element)
    for _, value in pairs(table) do
        if value == element then
            return true
        end
    end
    return false
end

function utils.isPullerRole(config)
    local pullerRoles = { "Puller", "Puller Tank", "Hunter" }
    local currentRole = config.items[config.current_item]
    return utils.contains(pullerRoles, currentRole)
end

function utils.checkAndExecuteFadeOverPulls(config)
    if not utils.isPullerRole(config) then
        if not config.hasPrintedPullerRoleMessage then
            print("Not in a puller role, skipping fade over pull action.")
            config.hasPrintedPullerRoleMessage = true
        end
        return
    else
        config.hasPrintedPullerRoleMessage = false
    end

    local xtAggroCount = mq.TLO.XAssist.XTFullHaterCount() or 0

    -- Debug print to track xtAggroCount
    --print("Current XTFullHaterCount: ", xtAggroCount)

    local haterCountThreshold = tonumber(config.xtHaterCountThreshold) or 0
    local currentTime = os.time()
    if xtAggroCount >= haterCountThreshold and (currentTime - lastAbilityUseTime >= abilityCooldown) then
        print("Threshold met. Current action: ", config.fadeAction)
        lastAbilityUseTime = currentTime

        if config.fadeAction == "SetFade" then
            print("NO ACTION SET! You Must Set A Fade Action!")
        elseif config.fadeAction == "Fade" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Escape;')
        elseif config.fadeAction == "FadedMemories" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /twist off; /aa act Fading Memories')
        elseif config.fadeAction == "FD" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Imitate Death;')
        elseif config.fadeAction == "PP" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Falsified Death;')
        elseif config.fadeAction == "DP" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Divine Peace;')
        elseif config.fadeAction == "VoU" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Veil of the Underbrush;')
        elseif config.fadeAction == "SS" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Self Stasis;')
        elseif config.fadeAction == "DoS" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Drape of Shadows;')
        elseif config.fadeAction == "FDSpell" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act 773;')
        elseif config.fadeAction == "BB" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Balefire Burst;')
        elseif config.fadeAction == "CT" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Cover Tracks;')
        elseif config.fadeAction == "IT" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Inconspicuous Totem;')
        elseif config.fadeAction == "SP" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act Self Preservation;')
        elseif config.fadeAction == "HiS" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act A Hole In Space;')
        elseif config.fadeAction == "HoW" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act A Howl of the Warlord;')
        elseif config.fadeAction == "FDSpellSK" then
            mq.cmd('/squelch /multiline ; /stopcast; /stick off; /attack off; /aa act 7756;')
        end
    --else
        --print("Threshold not met. Required: ", config.xtHaterCountThreshold, " Current: ", xtAggroCount) -- Debug print
    end
end

function utils.updateIniLoadedStatus(config)
    config.iniLoaded = config.selectedPath ~= ""
end

return utils
